// app/product/[id]/page.tsx

import { loadProductById, loadCategories, loadProducts, loadBrands, loadFacilities } from "@/lib/firebase";
import { notFound } from "next/navigation";
import ProductDetailClient from "./ProductDetailClient";
import Header from "@/components/Header";
import Footer from "@/components/Footer";
import Adsense from "@/components/Adsense";
import { ArrowLeft } from "lucide-react";
import Link from "next/link";
import { Category, Product } from "@/types";
import type { Metadata, ResolvingMetadata } from 'next';

// --- YENİ EKLENDİ: Dinamik Metadata Fonksiyonu ---
// Bu fonksiyon, her ürün sayfası için özel başlık (title) ve açıklama (description) oluşturur.
// Bu, SEO için çok önemlidir. Bu fonksiyon da 'async' olmalıdır.
type Props = {
  params: { id: string }
}

export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const id = params.id;
  const product = await loadProductById(id);

  if (!product) {
    return {
      title: 'Ürün Bulunamadı',
    }
  }

  return {
    title: `${product.name} | Vitamin & Takviye`,
    description: `${product.name} ürününün detaylı incelemesi, kullanıcı yorumları ve puanlaması.`,
    // openGraph, keywords gibi diğer metadata özelliklerini buraya ekleyebilirsiniz.
  }
}

// Bu fonksiyon, Next.js'e build sırasında hangi dinamik sayfaları oluşturacağını söyler.
// === DEĞİŞİKLİK BURADA ===
// Fonksiyonun başına 'async' anahtar kelimesi eklendi.
export default async function ProductPage({ params }: { params: { id:string } }) {
  const { id } = params;
  if (!id) {
    notFound();
  }

  // Gerekli tüm verileri paralel olarak çekiyoruz
  const [product, categories, allProducts, brands, facilities] = await Promise.all([
    loadProductById(id),
    loadCategories(),
    loadProducts(),
    loadBrands(),
    loadFacilities()
  ]).catch(error => {
      console.error("Failed to load data for product page:", error);
      notFound();
  });

  if (!product) {
    notFound();
  }

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      <Header
        categories={categories as Category[]}
        products={allProducts as Product[]}
        brands={brands}
        facilities={facilities}
        showCategories={false}
      />
      
      <main className="flex-1 w-full container mx-auto px-4 py-6">
        <div className="mb-4">
          <Link href="/" className="inline-flex items-center gap-2 text-blue-600 hover:underline">
            <ArrowLeft className="w-5 h-5" />
            Tüm Ürünlere Geri Dön
          </Link>
        </div>

        <ProductDetailClient product={product as Product} categories={categories as Category[]} />
        
        <div className="mt-8">
          <h3 className="text-center text-sm text-gray-400 mb-2">- Reklam -</h3>
          <Adsense 
            adClient="ca-pub-YOUR_ADSENSE_PUBLISHER_ID" 
            adSlot="YOUR_ADSENSE_AD_UNIT_ID" 
          />
        </div>
      </main>

      <Footer />
    </div>
  );
}